import os
import numpy as np
from scipy.interpolate import interp1d

def altcos(sza):
    """
    vals = altcos(sza)

    Drop-in replacement for cosine function for describing plasma production at the height of max plasma production
    using the Chapman function (which assumes the earth is round, not flat).

    The advantage of this approach is that the conductance gradient at the terminator is
    more realistic. This is important since conductance gradients appear in the equations that
    relate electric and magnetic fields. In addition, conductances above 90° sza are positive.

    The code is based on table lookup, and does not calculate the Chapman function.

    Author: S. M. Hatch (2024)

    Parameters
    ----------
    sza: 		array
        Solar zenith angle in degrees

    """
    shape = np.array(sza).shape

    PRODUCTIONFILE = os.path.join(os.path.dirname(
        __file__), 'chapman_euv_productionvalues.txt')

    MODELSZAS = np.arange(0, 120.1, 0.1)  # DO NOT CHANGE
    PRODUCTION = np.loadtxt(PRODUCTIONFILE)

    vals = interp1d(MODELSZAS,PRODUCTION,fill_value='extrapolate')(sza)
    
    return vals.reshape(shape)
    

def EUV_conductance(sza, F107=100, hallOrPed='hp',
                    calibration='MoenBrekke1993'):
    """
    cond = EUV_conductance(sza, F107, hallOrPed, calibration='MoenBrekke1993')

    Conductance calculated based on the plasma production at the height of max plasma production
    using the Chapman function (which assumes the earth is round, not flat) - and scaled to fit 
    your preferred empirical conductance model.

    The advantage of this approach is that the conductance gradient at the terminator is
    more realistic. This is important since conductance gradients appear in the equations that
    relate electric and magnetic fields. In addition, conductances above 90° sza are positive.

    The code is based on table lookup, and does not calculate the Chapman function.

    Author: S. M. Hatch (2021)

    Parameters
    ----------
    sza: 		array
        Solar zenith angle in degrees
    F107: float or array, optional
        F10.7 index - used to scale EUV conductance
        defualt is 100
    hallOrPed: 	string, optional
        Must be one of 'h', 'p', or 'hp', (corresponding to "Hall," "Pedersen," or both)
        default is both
    calibration: string, optional
        calibration to use in EUV_conductance calculation. Should be one of 
        'MoenBrekke1993', 'MoenBrekke1993_alt', 'Cousinsetal2015', with reference to
        the two papers cited below under References. 

        For Cousins et al (2015), Hall and Pedersen conductance are modeled using their 
        Equations (13) and (14).
        For Moen and Brekke (1993), Hall and Pedersen conductance are modeled using their 
        Equation (6).

    Returns
    -------
    If hall_or_pedersen == 'h':
        Hall conductances [mho] for each sza input value
    If hall_or_pedersen == 'p':
        Pedersen conductances [mho] for each sza input value
    If hall_or_pedersen == 'hp':
        Tuple of two arrays, one for Hall and one for Pedersen conductances [mho], for each sza input value


    Example
    -------
    # Get Hall conductance
    F107 = 70
    sza = np.arange(0,120.1,0.1)
    hall = EUV_conductance(sza,F107,'h')

    # Get Pedersen conductance
    F107 = 70
    sza = np.arange(0,120.1,0.1)
    pedersen = EUV_conductance(sza,F107,'p')

    # Get Hall and Pedersen conductance
    F107 = 70
    sza = np.arange(0,120.1,0.1)
    hall, pedersen = EUV_conductance(sza,F107,'hp')


    References
    ----------
    Cousins, E. D. P., Matsuo, T. and Richmond, A. D. (2015) ‘Mapping high-latitude ionospheric 
    electrodynamics with SuperDARN and AMPERE’, Journal of Geophysical Research: Space Physics, 
    120, pp. 5854–5870. doi: 10.1002/2014JA020463.

    Moen, J. and Brekke, A. (1993) ‘The solar flux influence on quiet time conductances in the 
    auroral ionosphere’, Geophysical Research Letters, 20(10), pp. 971–974. doi: 10.1029/92GL02109.

    """
    shape = np.array(sza).shape

    assert hallOrPed.lower() in [
        'h', 'p', 'hp'], "EUV_conductance: Must select one of 'h', 'p', or 'hp' for hallOrPed!"

    PRODUCTIONFILE = os.path.join(os.path.dirname(
        __file__), 'chapman_euv_productionvalues.txt')

    MODELSZAS = np.arange(0, 120.1, 0.1)  # DO NOT CHANGE
    PRODUCTION = np.loadtxt(PRODUCTIONFILE)

    getH = 'h' in hallOrPed.lower()
    getP = 'p' in hallOrPed.lower()

    if calibration not in ['MoenBrekke1993', 'MoenBrekke1993_alt', 'Cousinsetal2015']:
        defcal = 'MoenBrekke1993'
        print(f"Invalid calibration: {calibration}. Using {defcal}")
        calibration = defcal

    if calibration == 'MoenBrekke1993':
        # PedScl = 1.27               # Obtained by setting f107 = 1, sza = 0° in Moen and Brekke's (1993) empirical Pedersen conductance formula
        # HalScl = 1.35               # Obtained by setting f107 = 1, sza = 0° in Moen and Brekke's (1993) empirical Hall conductance formula

        f107pedexponent = 0.49
        f107hallexponent = 0.53

    elif calibration == 'MoenBrekke1993_alt':
        # Obtained by setting f107 = 1, sza = 0° in Moen and Brekke's (1993) empirical Pedersen conductance formula
        PedScl = 1.27
        # Obtained by setting f107 = 1, sza = 0° in Moen and Brekke's (1993) empirical Hall conductance formula
        HalScl = 1.35

        f107pedexponent = 0.49
        f107hallexponent = 0.53
        pedexponent = 0.65
        hallexponent = 0.79

    elif calibration == 'Cousinsetal2015':
        # Obtained by setting f107 = 1, sza = 0° in Brekke and Moen's (1993) empirical Pedersen conductance formula
        PedScl = 0.5
        # Obtained by setting f107 = 1, sza = 0° in Brekke and Moen's (1993) empirical Hall conductance formula
        HalScl = 1.8

        f107pedexponent = 0.667
        f107hallexponent = 0.5
        pedexponent = 0.667
        hallexponent = 1

    if calibration == 'MoenBrekke1993':

        if getH:
            halinterp = interp1d(MODELSZAS,
                                 F107**(f107hallexponent)*(0.81 *
                                                           PRODUCTION + 0.54*np.sqrt(PRODUCTION)),
                                 fill_value='extrapolate')
            sigh = halinterp(sza)  # moh

        if getP:
            pedinterp = interp1d(MODELSZAS,
                                 F107**(f107pedexponent)*(0.34 *
                                                          PRODUCTION + 0.93*np.sqrt(PRODUCTION)),
                                 fill_value='extrapolate')
            sigp = pedinterp(sza)  # moh

    else:

        if getH:
            halinterp = interp1d(MODELSZAS,
                                 F107**(f107hallexponent)*HalScl *
                                 (PRODUCTION)**(hallexponent),
                                 fill_value='extrapolate')
            sigh = halinterp(sza)  # moh

        if getP:
            pedinterp = interp1d(MODELSZAS,
                                 F107**(f107pedexponent)*PedScl *
                                 (PRODUCTION)**(pedexponent),
                                 fill_value='extrapolate')
            sigp = pedinterp(sza)  # moh

    if getH and getP:
        sigh[sigh < 0] = 0
        sigp[sigp < 0] = 0
        return sigh.reshape(shape), sigp.reshape(shape)
    elif getH:
        sigh[sigh < 0] = 0
        return sigh.reshape(shape)
    elif getP:
        sigp[sigp < 0] = 0
        return sigp.reshape(shape)

